const express = require('express');
const path = require('path');
const dotenv = require('dotenv');
const cookieParser = require('cookie-parser');
const morgan = require('morgan');
const session = require('express-session');
const flash = require('express-flash');
const { sequelize } = require('./models/db');

// Cargar variables de entorno
dotenv.config();

// Inicializar Express
const app = express();

// Configurar middlewares
app.use(express.json());
app.use(express.urlencoded({ extended: false }));
app.use(cookieParser());
app.use(morgan('dev'));

// Configurar express-session y flash messages
app.use(session({
  secret: process.env.SESSION_SECRET || 'nodolatam_secret',
  resave: false,
  saveUninitialized: false,
  cookie: { maxAge: 60 * 60 * 1000 } // 1 hora
}));
app.use(flash());

// Hacer disponibles los mensajes flash en todas las vistas
app.use((req, res, next) => {
  res.locals.success = req.flash('success');
  res.locals.error = req.flash('error');
  res.locals.info = req.flash('info');
  next();
});

// Configurar EJS como motor de plantillas
app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));

// Configurar express-ejs-layouts
require('./config/express-ejs-layouts')(app);

// Servir archivos estáticos
app.use(express.static(path.join(__dirname, 'public')));

// Middleware para añadir la ruta actual a todas las vistas
app.use((req, res, next) => {
  res.locals.path = req.originalUrl;
  next();
});

// Rutas
app.use('/', require('./routes/index'));
app.use('/auth', require('./routes/auth'));
app.use('/products', require('./routes/products'));
app.use('/categories', require('./routes/categories'));
app.use('/tags', require('./routes/tags'));
app.use('/profile', require('./routes/profile'));
app.use('/notifications', require('./routes/notifications'));
app.use('/admin', require('./routes/admin'));

// Página 404
app.use((req, res) => {
  res.status(404).render('404', {
    title: 'Página no encontrada',
    user: req.cookies.user ? JSON.parse(req.cookies.user) : null
  });
});

// Puerto
const PORT = process.env.PORT || 3000;

// Iniciar servidor
app.listen(PORT, async () => {
  console.log(`Servidor corriendo en el puerto ${PORT}`);
  
  try {
    await sequelize.authenticate();
    console.log('Conexión a la base de datos establecida correctamente.');
  } catch (error) {
    console.error('No se pudo conectar a la base de datos:', error);
  }
}); 