const { User, Product, Comment, Vote, Category, sequelize, Notification } = require('../models');
const { Op } = require('sequelize');
const bcrypt = require('bcryptjs');
const fs = require('fs');
const path = require('path');

// Dashboard principal
exports.dashboard = async (req, res) => {
  try {
    // Estadísticas generales
    const userCount = await User.count();
    const productCount = await Product.count();
    const commentCount = await Comment.count();
    const voteCount = await Vote.count();
    
    // Productos más votados
    const topProducts = await Product.findAll({
      order: [['votesCount', 'DESC']],
      limit: 5,
      include: [
        { model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] },
        { model: Category, as: 'category', attributes: ['id', 'name', 'slug'] }
      ]
    });
    
    // Usuarios más activos (por productos y comentarios)
    const topUsers = await User.findAll({
      attributes: {
        include: [
          [sequelize.literal('(SELECT COUNT(*) FROM Products WHERE Products.userId = User.id)'), 'productCount'],
          [sequelize.literal('(SELECT COUNT(*) FROM Comments WHERE Comments.userId = User.id)'), 'commentCount']
        ]
      },
      order: [
        [sequelize.literal('productCount + commentCount'), 'DESC']
      ],
      limit: 5
    });
    
    // Usuarios recientes
    const recentUsers = await User.findAll({
      order: [['createdAt', 'DESC']],
      limit: 5,
      attributes: ['id', 'name', 'email', 'avatar', 'createdAt']
    });
    
    // Actividad reciente
    const recentActivity = await Promise.all([
      // Últimos productos
      Product.findAll({
        order: [['createdAt', 'DESC']],
        limit: 5,
        include: [{ model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] }]
      }),
      
      // Últimos comentarios
      Comment.findAll({
        order: [['createdAt', 'DESC']],
        limit: 5,
        include: [
          { model: User, as: 'author', attributes: ['id', 'name', 'avatar'] },
          { model: Product, attributes: ['id', 'name'] }
        ]
      }),
      
      // Últimos votos
      Vote.findAll({
        order: [['createdAt', 'DESC']],
        limit: 5,
        include: [
          { model: User, attributes: ['id', 'name', 'avatar'] },
          { model: Product, attributes: ['id', 'name'] }
        ]
      })
    ]);
    
    const [recentProducts, recentComments, recentVotes] = recentActivity;
    
    res.render('admin/index', {
      title: 'Panel de Administración',
      layout: 'layouts/admin',
      path: '/admin',
      stats: { userCount, productCount, commentCount, voteCount },
      topProducts,
      topUsers,
      recentProducts,
      recentComments,
      recentVotes,
      recentUsers,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar el panel de administración',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Estadísticas
exports.stats = async (req, res) => {
  try {
    // Estadísticas generales
    const userCount = await User.count();
    const productCount = await Product.count();
    const commentCount = await Comment.count();
    const voteCount = await Vote.count();
    
    // Datos para gráficos (últimos 30 días)
    const thirtyDaysAgo = new Date();
    thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);
    
    // Función para obtener datos por día
    const getCountsByDay = async (model) => {
      const results = await model.findAll({
        attributes: [
          [sequelize.fn('DATE', sequelize.col('createdAt')), 'date'],
          [sequelize.fn('COUNT', sequelize.col('id')), 'count']
        ],
        where: {
          createdAt: {
            [Op.gte]: thirtyDaysAgo
          }
        },
        group: [sequelize.fn('DATE', sequelize.col('createdAt'))],
        order: [[sequelize.fn('DATE', sequelize.col('createdAt')), 'ASC']]
      });
      
      return results.map(result => ({
        date: result.dataValues.date,
        count: parseInt(result.dataValues.count)
      }));
    };
    
    // Obtener datos para cada modelo
    const [productsByDay, usersByDay, votesByDay] = await Promise.all([
      getCountsByDay(Product),
      getCountsByDay(User),
      getCountsByDay(Vote)
    ]);
    
    res.render('admin/stats', {
      title: 'Estadísticas',
      layout: 'layouts/admin',
      path: '/admin/stats',
      stats: { userCount, productCount, commentCount, voteCount },
      chartData: { productsByDay, usersByDay, votesByDay },
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar las estadísticas',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Gestión de usuarios
exports.getUsers = async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = 10;
    const offset = (page - 1) * limit;
    const search = req.query.search || '';
    
    let whereClause = {};
    if (search) {
      whereClause = {
        [Op.or]: [
          { name: { [Op.like]: `%${search}%` } },
          { email: { [Op.like]: `%${search}%` } }
        ]
      };
    }
    
    const { count, rows: users } = await User.findAndCountAll({
      where: whereClause,
      order: [['createdAt', 'DESC']],
      limit,
      offset
    });
    
    const totalPages = Math.ceil(count / limit);
    
    res.render('admin/users/index', {
      title: 'Gestión de Usuarios',
      layout: 'layouts/admin',
      path: '/admin/users',
      users,
      currentPage: page,
      totalPages,
      search,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar los usuarios',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

exports.getUser = async (req, res) => {
  try {
    const { id } = req.params;
    
    const user = await User.findByPk(id, {
      include: [
        {
          model: Product,
          as: 'products',
          limit: 5,
          order: [['createdAt', 'DESC']]
        }
      ]
    });
    
    if (!user) {
      req.flash('error', 'Usuario no encontrado');
      return res.redirect('/admin/users');
    }
    
    // Obtener estadísticas del usuario
    const productCount = await Product.count({ where: { userId: id } });
    const commentCount = await Comment.count({ where: { userId: id } });
    const voteCount = await Vote.count({ where: { userId: id } });
    
    res.render('admin/users/show', {
      title: `Usuario: ${user.name}`,
      layout: 'layouts/admin',
      path: '/admin/users',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null,
      profileUser: user,
      stats: { productCount, commentCount, voteCount }
    });
  } catch (error) {
    console.error(error);
    req.flash('error', 'Error al cargar el usuario');
    res.redirect('/admin/users');
  }
};

exports.updateUser = async (req, res) => {
  try {
    const { id } = req.params;
    const { name, email, role, isActive } = req.body;
    
    const user = await User.findByPk(id);
    
    if (!user) {
      return res.status(404).json({
        success: false,
        message: 'Usuario no encontrado'
      });
    }
    
    // Actualizar datos del usuario
    await user.update({
      name,
      email,
      role: role || 'user',
      isActive: isActive === 'true'
    });
    
    res.json({
      success: true,
      message: 'Usuario actualizado correctamente'
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      success: false,
      message: 'Error al actualizar el usuario'
    });
  }
};

exports.deleteUser = async (req, res) => {
  try {
    const { id } = req.params;
    
    const user = await User.findByPk(id);
    
    if (!user) {
      return res.status(404).json({
        success: false,
        message: 'Usuario no encontrado'
      });
    }
    
    // Eliminar avatar asociado
    if (user.avatar) {
      const avatarPath = path.join(__dirname, '../public/uploads/avatars', user.avatar);
      if (fs.existsSync(avatarPath)) {
        fs.unlinkSync(avatarPath);
      }
    }
    
    // Eliminar el usuario
    await user.destroy();
    
    res.json({
      success: true,
      message: 'Usuario eliminado correctamente'
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      success: false,
      message: 'Error al eliminar el usuario'
    });
  }
};

// Gestión de productos
exports.getProducts = async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = 10;
    const offset = (page - 1) * limit;
    const search = req.query.search || '';
    
    let whereClause = {};
    if (search) {
      whereClause = {
        [Op.or]: [
          { name: { [Op.like]: `%${search}%` } },
          { description: { [Op.like]: `%${search}%` } }
        ]
      };
    }
    
    const { count, rows: products } = await Product.findAndCountAll({
      where: whereClause,
      include: [
        { model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] },
        { model: Category, as: 'category', attributes: ['id', 'name', 'slug'] }
      ],
      order: [['createdAt', 'DESC']],
      limit,
      offset
    });
    
    const totalPages = Math.ceil(count / limit);
    
    res.render('admin/products/index', {
      title: 'Gestión de Productos',
      layout: 'layouts/admin',
      path: '/admin/products',
      products,
      currentPage: page,
      totalPages,
      search,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar los productos',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

exports.getProduct = async (req, res) => {
  try {
    const { id } = req.params;
    
    const product = await Product.findByPk(id, {
      include: [
        { model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] },
        { model: Category, as: 'category', attributes: ['id', 'name', 'slug'] },
        { 
          model: Comment,
          include: [
            { model: User, as: 'author', attributes: ['id', 'name', 'avatar'] }
          ],
          order: [['createdAt', 'DESC']],
          limit: 10
        }
      ]
    });
    
    if (!product) {
      req.flash('error', 'Producto no encontrado');
      return res.redirect('/admin/products');
    }
    
    // Obtener categorías para el select de edición
    const categories = await Category.findAll({
      order: [['name', 'ASC']]
    });
    
    res.render('admin/products/show', {
      title: `Producto: ${product.name}`,
      layout: 'layouts/admin',
      path: '/admin/products',
      product,
      categories,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    req.flash('error', 'Error al cargar el producto');
    res.redirect('/admin/products');
  }
};

exports.updateProduct = async (req, res) => {
  try {
    const { id } = req.params;
    const { name, description, url, categoryId } = req.body;
    
    const product = await Product.findByPk(id);
    
    if (!product) {
      return res.status(404).json({
        success: false,
        message: 'Producto no encontrado'
      });
    }
    
    // Actualizar producto
    await product.update({
      name,
      description,
      url,
      categoryId: categoryId || null
    });
    
    res.json({
      success: true,
      message: 'Producto actualizado correctamente'
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      success: false,
      message: 'Error al actualizar el producto'
    });
  }
};

exports.deleteProduct = async (req, res) => {
  try {
    const { id } = req.params;
    
    const product = await Product.findByPk(id);
    
    if (!product) {
      return res.status(404).json({
        success: false,
        message: 'Producto no encontrado'
      });
    }
    
    // Eliminar imagen del producto
    if (product.image && product.image !== 'default-product.png') {
      const imagePath = path.join(__dirname, '../public/uploads/products', product.image);
      if (fs.existsSync(imagePath)) {
        fs.unlinkSync(imagePath);
      }
    }
    
    // Eliminar producto (los comentarios y votos se eliminarán en cascada)
    await product.destroy();
    
    res.json({
      success: true,
      message: 'Producto eliminado correctamente'
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      success: false,
      message: 'Error al eliminar el producto'
    });
  }
};

// Gestión de notificaciones
exports.getNotifications = async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = 20;
    const offset = (page - 1) * limit;
    const search = req.query.search || '';
    
    let whereClause = {};
    if (search) {
      whereClause = {
        message: { [Op.like]: `%${search}%` }
      };
    }
    
    const { count, rows: notifications } = await Notification.findAndCountAll({
      where: whereClause,
      include: [
        { model: User, as: 'user', attributes: ['id', 'name', 'avatar'] }
      ],
      order: [['createdAt', 'DESC']],
      limit,
      offset
    });
    
    const totalPages = Math.ceil(count / limit);
    
    res.render('admin/notifications/index', {
      title: 'Gestión de Notificaciones',
      notifications,
      currentPage: page,
      totalPages,
      search,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar las notificaciones',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Crear notificación para todos los usuarios
exports.createNotification = async (req, res) => {
  try {
    const { message, link } = req.body;
    
    if (!message) {
      req.flash('error', 'El mensaje es obligatorio');
      return res.redirect('/admin/notifications');
    }
    
    // Obtener todos los usuarios
    const users = await User.findAll();
    
    // Crear notificación para cada usuario
    for (const user of users) {
      await Notification.create({
        userId: user.id,
        type: 'system',
        message,
        link: link || null
      });
    }
    
    req.flash('success', 'Notificación enviada a todos los usuarios');
    res.redirect('/admin/notifications');
  } catch (error) {
    console.error(error);
    req.flash('error', 'Error al crear la notificación');
    res.redirect('/admin/notifications');
  }
};

// Eliminar notificación (admin)
exports.deleteNotification = async (req, res) => {
  try {
    const { id } = req.params;
    
    const notification = await Notification.findByPk(id);
    
    if (!notification) {
      return res.status(404).json({
        success: false,
        message: 'Notificación no encontrada'
      });
    }
    
    await notification.destroy();
    
    res.json({
      success: true,
      message: 'Notificación eliminada correctamente'
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      success: false,
      message: 'Error al eliminar la notificación'
    });
  }
}; 