const { validationResult } = require('express-validator');
const jwt = require('jsonwebtoken');
const { User } = require('../models');

// Registrar un nuevo usuario
exports.register = async (req, res) => {
  // Verificar errores de validación
  const errors = validationResult(req);
  if (!errors.isEmpty()) {
    return res.status(400).render('auth/register', {
      title: 'Registro',
      errors: errors.array(),
      user: null,
      data: req.body
    });
  }

  const { name, email, password } = req.body;

  try {
    // Verificar si el usuario ya existe
    const existingUser = await User.findOne({ where: { email } });
    if (existingUser) {
      return res.status(400).render('auth/register', {
        title: 'Registro',
        errors: [{ msg: 'El correo electrónico ya está registrado' }],
        user: null,
        data: req.body
      });
    }

    // Crear nuevo usuario
    const user = await User.create({
      name,
      email,
      password
    });

    // Generar JWT
    const token = jwt.sign(
      { id: user.id },
      process.env.JWT_SECRET,
      { expiresIn: '1d' }
    );

    // Guardar usuario en cookie
    const userInfo = {
      id: user.id,
      name: user.name,
      email: user.email,
      avatar: user.avatar,
      isAdmin: user.isAdmin
    };
    
    res.cookie('token', token, {
      httpOnly: true,
      maxAge: 24 * 60 * 60 * 1000 // 1 día
    });
    
    res.cookie('user', JSON.stringify(userInfo), {
      maxAge: 24 * 60 * 60 * 1000 // 1 día
    });

    // Redireccionar al inicio
    res.redirect('/');
  } catch (error) {
    console.error(error);
    res.status(500).render('auth/register', {
      title: 'Registro',
      errors: [{ msg: 'Error en el servidor. Intente nuevamente.' }],
      user: null,
      data: req.body
    });
  }
};

// Iniciar sesión
exports.login = async (req, res) => {
  // Verificar errores de validación
  const errors = validationResult(req);
  if (!errors.isEmpty()) {
    return res.status(400).render('auth/login', {
      title: 'Iniciar Sesión',
      errors: errors.array(),
      user: null,
      data: req.body
    });
  }

  const { email, password } = req.body;

  try {
    // Buscar usuario
    const user = await User.findOne({ where: { email } });
    if (!user) {
      return res.status(400).render('auth/login', {
        title: 'Iniciar Sesión',
        errors: [{ msg: 'Credenciales inválidas' }],
        user: null,
        data: req.body
      });
    }

    // Verificar contraseña
    const isMatch = await user.comparePassword(password);
    if (!isMatch) {
      return res.status(400).render('auth/login', {
        title: 'Iniciar Sesión',
        errors: [{ msg: 'Credenciales inválidas' }],
        user: null,
        data: req.body
      });
    }

    // Generar JWT
    const token = jwt.sign(
      { id: user.id },
      process.env.JWT_SECRET,
      { expiresIn: '1d' }
    );

    // Guardar usuario en cookie
    const userInfo = {
      id: user.id,
      name: user.name,
      email: user.email,
      avatar: user.avatar,
      isAdmin: user.isAdmin
    };
    
    res.cookie('token', token, {
      httpOnly: true,
      maxAge: 24 * 60 * 60 * 1000 // 1 día
    });
    
    res.cookie('user', JSON.stringify(userInfo), {
      maxAge: 24 * 60 * 60 * 1000 // 1 día
    });

    // Redireccionar al inicio
    res.redirect('/');
  } catch (error) {
    console.error(error);
    res.status(500).render('auth/login', {
      title: 'Iniciar Sesión',
      errors: [{ msg: 'Error en el servidor. Intente nuevamente.' }],
      user: null,
      data: req.body
    });
  }
};

// Cerrar sesión
exports.logout = (req, res) => {
  res.clearCookie('token');
  res.clearCookie('user');
  res.redirect('/');
};

// Mostrar formulario de registro
exports.showRegisterForm = (req, res) => {
  res.render('auth/register', {
    title: 'Registro',
    errors: [],
    user: req.cookies.user ? JSON.parse(req.cookies.user) : null,
    data: {}
  });
};

// Mostrar formulario de login
exports.showLoginForm = (req, res) => {
  res.render('auth/login', {
    title: 'Iniciar Sesión',
    errors: [],
    user: req.cookies.user ? JSON.parse(req.cookies.user) : null,
    data: {}
  });
}; 