const { Category, Product, User } = require('../models');
const { validationResult } = require('express-validator');
const { Op } = require('sequelize');
const slugify = require('slugify');

// Obtener todas las categorías
exports.getCategories = async (req, res) => {
  try {
    const categories = await Category.findAll({
      order: [['name', 'ASC']]
    });
    
    res.render('categories/index', {
      title: 'Categorías',
      categories,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar las categorías',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Obtener productos por categoría
exports.getCategoryProducts = async (req, res) => {
  try {
    const { slug } = req.params;
    
    const category = await Category.findOne({
      where: { slug }
    });
    
    if (!category) {
      return res.status(404).render('404', {
        title: 'Categoría no encontrada',
        user: req.cookies.user ? JSON.parse(req.cookies.user) : null
      });
    }
    
    const page = parseInt(req.query.page) || 1;
    const limit = 10;
    const offset = (page - 1) * limit;
    const sortBy = req.query.sort || 'date'; // 'date' o 'votes'
    
    let order;
    if (sortBy === 'votes') {
      order = [['votesCount', 'DESC']];
    } else {
      order = [['createdAt', 'DESC']];
    }
    
    const { count, rows: products } = await Product.findAndCountAll({
      where: { categoryId: category.id },
      include: [
        { model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] },
        { model: Category, as: 'category', attributes: ['id', 'name', 'slug'] }
      ],
      order,
      limit,
      offset
    });
    
    const totalPages = Math.ceil(count / limit);
    
    res.render('categories/show', {
      title: `Categoría: ${category.name}`,
      category,
      products,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null,
      currentPage: page,
      totalPages,
      sortBy
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar los productos de la categoría',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// ADMIN: Listar categorías
exports.adminGetCategories = async (req, res) => {
  try {
    const categories = await Category.findAll({
      order: [['name', 'ASC']],
      include: [
        {
          model: Product,
          attributes: ['id']
        }
      ]
    });
    
    // Contar productos por categoría
    const categoriesWithCount = categories.map(category => {
      const count = category.Products ? category.Products.length : 0;
      return {
        ...category.toJSON(),
        productCount: count
      };
    });
    
    res.render('admin/categories/index', {
      title: 'Gestión de Categorías',
      layout: 'layouts/admin',
      path: '/admin/categories',
      categories: categoriesWithCount,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar las categorías',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// ADMIN: Mostrar formulario para crear categoría
exports.adminShowCreateForm = (req, res) => {
  res.render('admin/categories/create', {
    title: 'Crear Categoría',
    layout: 'layouts/admin',
    path: '/admin/categories',
    errors: [],
    data: {},
    user: req.cookies.user ? JSON.parse(req.cookies.user) : null
  });
};

// ADMIN: Crear categoría
exports.adminCreateCategory = async (req, res) => {
  // Verificar errores de validación
  const errors = validationResult(req);
  if (!errors.isEmpty()) {
    return res.status(400).render('admin/categories/create', {
      title: 'Crear Categoría',
      layout: 'layouts/admin',
      path: '/admin/categories',
      errors: errors.array(),
      data: req.body,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }

  try {
    const { name, description, icon } = req.body;
    
    // Crear slug desde el nombre
    const slug = slugify(name, { 
      lower: true,
      strict: true,
      locale: 'es'
    });
    
    // Verificar si ya existe una categoría con ese slug
    const existingCategory = await Category.findOne({ where: { slug } });
    if (existingCategory) {
      return res.status(400).render('admin/categories/create', {
        title: 'Crear Categoría',
        layout: 'layouts/admin',
        path: '/admin/categories',
        errors: [{ msg: 'Ya existe una categoría con un nombre similar' }],
        data: req.body,
        user: req.cookies.user ? JSON.parse(req.cookies.user) : null
      });
    }
    
    // Crear categoría
    const newCategory = await Category.create({
      name,
      slug,
      description: description || '',
      icon: icon || 'fa-solid fa-folder'
    });
    
    if (!newCategory) {
      throw new Error('No se pudo crear la categoría');
    }
    
    req.flash('success', 'Categoría creada correctamente');
    return res.redirect('/admin/categories');
  } catch (error) {
    console.error('Error al crear categoría:', error);
    
    return res.status(500).render('admin/categories/create', {
      title: 'Crear Categoría',
      layout: 'layouts/admin',
      path: '/admin/categories',
      errors: [{ msg: 'Error al crear la categoría: ' + (error.message || 'Error desconocido') }],
      data: req.body,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// ADMIN: Mostrar formulario para editar categoría
exports.adminShowEditForm = async (req, res) => {
  try {
    const { id } = req.params;
    
    const category = await Category.findByPk(id);
    
    if (!category) {
      req.flash('error', 'Categoría no encontrada');
      return res.redirect('/admin/categories');
    }
    
    res.render('admin/categories/edit', {
      title: 'Editar Categoría',
      layout: 'layouts/admin',
      path: '/admin/categories',
      category,
      errors: [],
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    req.flash('error', 'Error al cargar la categoría');
    res.redirect('/admin/categories');
  }
};

// ADMIN: Actualizar categoría
exports.adminUpdateCategory = async (req, res) => {
  // Verificar errores de validación
  const errors = validationResult(req);
  if (!errors.isEmpty()) {
    return res.status(400).json({
      success: false,
      errors: errors.array()
    });
  }

  try {
    const { id } = req.params;
    const { name, description, icon } = req.body;
    
    const category = await Category.findByPk(id);
    
    if (!category) {
      return res.status(404).json({
        success: false,
        message: 'Categoría no encontrada'
      });
    }
    
    // Crear slug desde el nombre si cambió
    let slug = category.slug;
    if (name !== category.name) {
      slug = slugify(name, { 
        lower: true,
        strict: true,
        locale: 'es'
      });
      
      // Verificar si ya existe otra categoría con ese slug
      const existingCategory = await Category.findOne({ 
        where: { 
          slug,
          id: { [Op.ne]: id } // No incluir la categoría actual
        } 
      });
      
      if (existingCategory) {
        return res.status(400).json({
          success: false,
          message: 'Ya existe una categoría con un nombre similar'
        });
      }
    }
    
    // Actualizar categoría
    await category.update({
      name,
      slug,
      description,
      icon: icon || 'fa-solid fa-folder'
    });
    
    res.json({
      success: true,
      message: 'Categoría actualizada correctamente'
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      success: false,
      message: 'Error al actualizar la categoría'
    });
  }
};

// ADMIN: Eliminar categoría
exports.adminDeleteCategory = async (req, res) => {
  try {
    const { id } = req.params;
    
    const category = await Category.findByPk(id);
    
    if (!category) {
      return res.status(404).json({
        success: false,
        message: 'Categoría no encontrada'
      });
    }
    
    // Contar productos asociados
    const productCount = await Product.count({ where: { categoryId: id } });
    
    if (productCount > 0) {
      return res.status(400).json({
        success: false,
        message: `No se puede eliminar la categoría porque tiene ${productCount} productos asociados`
      });
    }
    
    await category.destroy();
    
    res.json({
      success: true,
      message: 'Categoría eliminada correctamente'
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      success: false,
      message: 'Error al eliminar la categoría'
    });
  }
}; 