const { Product, User, Category, Tag } = require('../models');

// Página de inicio
exports.getHomePage = async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = 10;
    const offset = (page - 1) * limit;
    const sortBy = req.query.sort || 'date'; // 'date' o 'votes'
    
    let order;
    if (sortBy === 'votes') {
      order = [['votesCount', 'DESC']];
    } else {
      order = [['createdAt', 'DESC']];
    }
    
    // Si hay parámetros de consulta, mostrar la vista normal con paginación
    if (req.query.page || req.query.sort) {
      const { count, rows: products } = await Product.findAndCountAll({
        include: [
          { model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] }
        ],
        order,
        limit,
        offset
      });
      
      const totalPages = Math.ceil(count / limit);
      
      return res.render('home', {
        title: 'Inicio',
        products,
        user: req.cookies.user ? JSON.parse(req.cookies.user) : null,
        currentPage: page,
        totalPages,
        sortBy
      });
    }
    
    // Si no hay parámetros, mostrar la landing page
    // Obtener productos populares (los 6 más votados)
    const popularProducts = await Product.findAll({
      include: [
        { model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] }
      ],
      order: [['votesCount', 'DESC']],
      limit: 6
    });
    
    // Obtener productos recientes (los 6 más recientes)
    const recentProducts = await Product.findAll({
      include: [
        { model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] }
      ],
      order: [['createdAt', 'DESC']],
      limit: 6
    });
    
    // Obtener categorías destacadas
    const categories = await Category.findAll({
      limit: 6,
      order: [['name', 'ASC']]
    });
    
    res.render('landing', {
      title: 'Bienvenido a nodolatam.dev',
      popularProducts,
      recentProducts,
      categories,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar la página de inicio',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
}; 