const { Notification, User, Product } = require('../models');

// Crear una notificación
exports.createNotification = async (userId, type, message, link = null) => {
  try {
    await Notification.create({
      userId,
      type,
      message,
      link
    });
    return true;
  } catch (error) {
    console.error('Error al crear notificación:', error);
    return false;
  }
};

// Obtener notificaciones de un usuario
exports.getUserNotifications = async (req, res) => {
  try {
    if (!req.cookies.user) {
      return res.status(401).json({
        success: false,
        message: 'No autorizado'
      });
    }

    const user = JSON.parse(req.cookies.user);
    
    const notifications = await Notification.findAll({
      where: { userId: user.id },
      order: [['createdAt', 'DESC']],
      limit: 10
    });
    
    // Contar notificaciones no leídas
    const unreadCount = await Notification.count({
      where: { 
        userId: user.id,
        isRead: false
      }
    });
    
    res.json({
      success: true,
      notifications,
      unreadCount
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      success: false,
      message: 'Error al obtener notificaciones'
    });
  }
};

// Marcar notificación como leída
exports.markAsRead = async (req, res) => {
  try {
    const { id } = req.params;
    const user = JSON.parse(req.cookies.user);
    
    const notification = await Notification.findOne({
      where: { 
        id,
        userId: user.id
      }
    });
    
    if (!notification) {
      return res.status(404).json({
        success: false,
        message: 'Notificación no encontrada'
      });
    }
    
    await notification.update({ isRead: true });
    
    res.json({
      success: true,
      message: 'Notificación marcada como leída'
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      success: false,
      message: 'Error al marcar la notificación como leída'
    });
  }
};

// Marcar todas las notificaciones como leídas
exports.markAllAsRead = async (req, res) => {
  try {
    const user = JSON.parse(req.cookies.user);
    
    await Notification.update(
      { isRead: true },
      { where: { userId: user.id } }
    );
    
    res.json({
      success: true,
      message: 'Todas las notificaciones marcadas como leídas'
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      success: false,
      message: 'Error al marcar las notificaciones como leídas'
    });
  }
};

// Eliminar notificación
exports.deleteNotification = async (req, res) => {
  try {
    const { id } = req.params;
    const user = JSON.parse(req.cookies.user);
    
    const notification = await Notification.findOne({
      where: { 
        id,
        userId: user.id
      }
    });
    
    if (!notification) {
      return res.status(404).json({
        success: false,
        message: 'Notificación no encontrada'
      });
    }
    
    await notification.destroy();
    
    res.json({
      success: true,
      message: 'Notificación eliminada'
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      success: false,
      message: 'Error al eliminar la notificación'
    });
  }
}; 