const { validationResult } = require('express-validator');
const fs = require('fs');
const path = require('path');
const { Product, User, Comment, Vote, sequelize, Category, Tag } = require('../models');
const { Op } = require('sequelize');
const notificationController = require('./notificationController');

// Mostrar todos los productos
exports.getProducts = async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = 10;
    const offset = (page - 1) * limit;
    const sortBy = req.query.sort || 'date'; // 'date' o 'votes'
    
    let order;
    if (sortBy === 'votes') {
      order = [['votesCount', 'DESC']];
    } else {
      order = [['createdAt', 'DESC']];
    }
    
    const { count, rows: products } = await Product.findAndCountAll({
      include: [
        { model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] }
      ],
      order,
      limit,
      offset
    });
    
    const totalPages = Math.ceil(count / limit);
    
    res.render('products/index', {
      title: 'Productos',
      products,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null,
      currentPage: page,
      totalPages,
      sortBy
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar los productos',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Mostrar un producto específico
exports.getProduct = async (req, res) => {
  try {
    const { id } = req.params;
    
    const product = await Product.findByPk(id, {
      include: [
        { model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] },
        { model: Category, as: 'category', attributes: ['id', 'name', 'slug'] },
        { 
          model: Tag,
          as: 'tags',
          attributes: ['id', 'name', 'slug'],
          through: { attributes: [] }
        },
        { 
          model: Comment,
          include: [
            { model: User, as: 'author', attributes: ['id', 'name', 'avatar'] }
          ],
          order: [['createdAt', 'DESC']]
        }
      ]
    });
    
    if (!product) {
      return res.status(404).render('404', {
        title: 'Producto no encontrado',
        user: req.cookies.user ? JSON.parse(req.cookies.user) : null
      });
    }
    
    // Verificar si el usuario ha votado por este producto
    let hasVoted = false;
    if (req.cookies.user) {
      const user = JSON.parse(req.cookies.user);
      const vote = await Vote.findOne({
        where: {
          userId: user.id,
          productId: id
        }
      });
      
      hasVoted = !!vote;
    }
    
    res.render('products/show', {
      title: product.name,
      product,
      hasVoted,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar el producto',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Mostrar formulario para crear producto
exports.showCreateForm = async (req, res) => {
  try {
    // Obtener categorías para el select
    const categories = await Category.findAll({
      order: [['name', 'ASC']]
    });
    
    // Obtener etiquetas para el select múltiple
    const tags = await Tag.findAll({
      order: [['name', 'ASC']]
    });
    
    res.render('products/create', {
      title: 'Crear Producto',
      categories,
      tags,
      errors: [],
      data: {},
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar el formulario',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Crear producto
exports.createProduct = async (req, res) => {
  // Verificar errores de validación
  const errors = validationResult(req);
  
  try {
    // Obtener categorías para el select (en caso de error)
    const categories = await Category.findAll({
      order: [['name', 'ASC']]
    });
    
    // Obtener etiquetas para el select múltiple (en caso de error)
    const tags = await Tag.findAll({
      order: [['name', 'ASC']]
    });
    
    if (!errors.isEmpty()) {
      return res.status(400).render('products/create', {
        title: 'Crear Producto',
        categories,
        tags,
        errors: errors.array(),
        data: req.body,
        user: req.cookies.user ? JSON.parse(req.cookies.user) : null
      });
    }

    // Verificar si se subió una imagen
    if (!req.file) {
      return res.status(400).render('products/create', {
        title: 'Crear Producto',
        categories,
        tags,
        errors: [{ msg: 'La imagen es obligatoria' }],
        data: req.body,
        user: req.cookies.user ? JSON.parse(req.cookies.user) : null
      });
    }

    const { name, description, url, categoryId, pricingType, highlights, twitterUrl, githubUrl } = req.body;
    const user = JSON.parse(req.cookies.user);
    
    // Obtener etiquetas seleccionadas (si las hay)
    const selectedTags = Array.isArray(req.body.tags) ? req.body.tags : req.body.tags ? [req.body.tags] : [];

    // Crear producto
    const product = await Product.create({
      name,
      description,
      url,
      image: req.file.filename,
      userId: user.id,
      categoryId: categoryId || null,
      pricingType: pricingType || 'free',
      highlights: highlights || '',
      twitterUrl: twitterUrl || null,
      githubUrl: githubUrl || null
    });
    
    // Asignar etiquetas al producto
    if (selectedTags.length > 0) {
      await product.setTags(selectedTags);
    }

    // Redireccionar al inicio
    req.flash('success', 'Producto creado correctamente');
    res.redirect('/');
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al crear el producto',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Votar por un producto
exports.voteProduct = async (req, res) => {
  try {
    const { id } = req.params;
    const user = JSON.parse(req.cookies.user);
    
    // Verificar si el producto existe
    const product = await Product.findByPk(id, {
      include: [{ model: User, as: 'creator' }]
    });
    
    if (!product) {
      return res.status(404).json({ success: false, message: 'Producto no encontrado' });
    }
    
    // Verificar si el usuario ya votó
    const existingVote = await Vote.findOne({
      where: {
        userId: user.id,
        productId: id
      }
    });
    
    if (existingVote) {
      return res.status(400).json({ success: false, message: 'Ya has votado por este producto' });
    }
    
    // Crear voto y actualizar contador
    await sequelize.transaction(async (t) => {
      await Vote.create({
        userId: user.id,
        productId: id
      }, { transaction: t });
      
      await Product.update(
        { votesCount: sequelize.literal('votesCount + 1') },
        { where: { id }, transaction: t }
      );
    });
    
    // Obtener el producto actualizado
    const updatedProduct = await Product.findByPk(id);
    
    // Crear notificación para el creador del producto (si no es el mismo usuario)
    if (product.creator.id !== user.id) {
      await notificationController.createNotification(
        product.creator.id,
        'vote',
        `${user.name} ha votado por tu producto "${product.name}"`,
        `/products/${id}`
      );
    }
    
    res.json({
      success: true,
      message: 'Voto registrado',
      votesCount: updatedProduct.votesCount
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ success: false, message: 'Error al procesar el voto' });
  }
};

// Comentar un producto
exports.commentProduct = async (req, res) => {
  // Verificar errores de validación
  const errors = validationResult(req);
  if (!errors.isEmpty()) {
    return res.status(400).json({
      success: false,
      errors: errors.array()
    });
  }

  try {
    const { id } = req.params;
    const { text } = req.body;
    const user = JSON.parse(req.cookies.user);
    
    // Verificar si el producto existe
    const product = await Product.findByPk(id, {
      include: [{ model: User, as: 'creator' }]
    });
    
    if (!product) {
      return res.status(404).json({ success: false, message: 'Producto no encontrado' });
    }
    
    // Crear comentario
    const comment = await Comment.create({
      text,
      userId: user.id,
      productId: id
    });
    
    // Obtener el comentario con los datos del autor
    const newComment = await Comment.findByPk(comment.id, {
      include: [{ model: User, as: 'author', attributes: ['id', 'name', 'avatar'] }]
    });
    
    // Crear notificación para el creador del producto (si no es el mismo usuario)
    if (product.creator.id !== user.id) {
      await notificationController.createNotification(
        product.creator.id,
        'comment',
        `${user.name} ha comentado en tu producto "${product.name}"`,
        `/products/${id}`
      );
    }
    
    res.json({
      success: true,
      comment: newComment
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ success: false, message: 'Error al crear el comentario' });
  }
};

// Obtener productos de un usuario específico
exports.getUserProducts = async (req, res) => {
  try {
    const { userId } = req.params;
    const page = parseInt(req.query.page) || 1;
    const limit = 12;
    const offset = (page - 1) * limit;
    
    // Obtener información del usuario
    const profileUser = await User.findByPk(userId, {
      attributes: ['id', 'name', 'avatar', 'createdAt']
    });
    
    if (!profileUser) {
      return res.status(404).render('404', {
        title: 'Usuario no encontrado',
        user: req.cookies.user ? JSON.parse(req.cookies.user) : null
      });
    }
    
    // Obtener productos del usuario
    const { count, rows: products } = await Product.findAndCountAll({
      where: { userId },
      include: [
        { model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] },
        { model: Category, as: 'category', attributes: ['id', 'name', 'slug'] },
        { 
          model: Tag,
          as: 'tags',
          attributes: ['id', 'name', 'slug'],
          through: { attributes: [] }
        }
      ],
      order: [['createdAt', 'DESC']],
      limit,
      offset
    });
    
    const totalPages = Math.ceil(count / limit);
    
    res.render('products/user', {
      title: `Productos de ${profileUser.name}`,
      products,
      profileUser,
      currentPage: page,
      totalPages,
      isOwnProfile: req.cookies.user ? JSON.parse(req.cookies.user).id === parseInt(userId) : false,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar los productos del usuario',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Buscar productos
exports.searchProducts = async (req, res) => {
  try {
    const { q } = req.query;
    
    if (!q) {
      return res.redirect('/');
    }
    
    const products = await Product.findAll({
      where: {
        [Op.or]: [
          { name: { [Op.like]: `%${q}%` } },
          { description: { [Op.like]: `%${q}%` } }
        ]
      },
      include: [
        { model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] }
      ],
      order: [['createdAt', 'DESC']]
    });
    
    res.render('products/search', {
      title: `Búsqueda: ${q}`,
      products,
      query: q,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al buscar productos',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Obtener votos de un usuario específico
exports.getUserVotes = async (req, res) => {
  try {
    const { userId } = req.params;
    const page = parseInt(req.query.page) || 1;
    const limit = 12;
    const offset = (page - 1) * limit;
    
    // Obtener información del usuario
    const profileUser = await User.findByPk(userId, {
      attributes: ['id', 'name', 'avatar', 'createdAt']
    });
    
    if (!profileUser) {
      return res.status(404).render('404', {
        title: 'Usuario no encontrado',
        user: req.cookies.user ? JSON.parse(req.cookies.user) : null
      });
    }
    
    // Obtener los votos del usuario con los productos asociados
    const { count, rows: votes } = await Vote.findAndCountAll({
      where: { userId },
      include: [
        { 
          model: Product,
          include: [
            { model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] },
            { model: Category, as: 'category', attributes: ['id', 'name', 'slug'] },
            { 
              model: Tag,
              as: 'tags',
              attributes: ['id', 'name', 'slug'],
              through: { attributes: [] }
            }
          ]
        }
      ],
      order: [['createdAt', 'DESC']],
      limit,
      offset
    });
    
    const totalPages = Math.ceil(count / limit);
    
    res.render('products/votes', {
      title: `Votos de ${profileUser.name}`,
      votes,
      profileUser,
      currentPage: page,
      totalPages,
      isOwnProfile: req.cookies.user ? JSON.parse(req.cookies.user).id === parseInt(userId) : false,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar los votos del usuario',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// ADMIN: Obtener producto individual para administración
exports.adminGetProduct = async (req, res) => {
  try {
    const { id } = req.params;
    
    const product = await Product.findByPk(id, {
      include: [
        { model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] },
        { model: Category, as: 'category', attributes: ['id', 'name', 'slug'] },
        { 
          model: Tag,
          as: 'tags',
          attributes: ['id', 'name', 'slug'],
          through: { attributes: [] }
        },
        { 
          model: Comment,
          include: [
            { model: User, as: 'author', attributes: ['id', 'name', 'avatar'] }
          ],
          order: [['createdAt', 'DESC']]
        }
      ]
    });
    
    if (!product) {
      req.flash('error', 'Producto no encontrado');
      return res.redirect('/admin/products');
    }
    
    // Obtener todas las etiquetas para el formulario de edición
    const allTags = await Tag.findAll({
      order: [['name', 'ASC']]
    });
    
    res.render('admin/products/show', {
      title: `Producto: ${product.name}`,
      product,
      allTags,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    req.flash('error', 'Error al cargar el producto');
    res.redirect('/admin/products');
  }
};

// ADMIN: Actualizar producto
exports.adminUpdateProduct = async (req, res) => {
  try {
    const { id } = req.params;
    const { name, description, url, categoryId, tags } = req.body;
    
    const product = await Product.findByPk(id);
    
    if (!product) {
      return res.status(404).json({
        success: false,
        message: 'Producto no encontrado'
      });
    }
    
    // Actualizar producto
    await product.update({
      name,
      description,
      url,
      categoryId: categoryId || null
    });
    
    // Actualizar etiquetas
    const selectedTags = Array.isArray(tags) ? tags : tags ? [tags] : [];
    await product.setTags(selectedTags);
    
    res.json({
      success: true,
      message: 'Producto actualizado correctamente'
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      success: false,
      message: 'Error al actualizar el producto'
    });
  }
};

// Mostrar formulario para crear herramienta de IA
exports.showCreateAIToolForm = async (req, res) => {
  try {
    // Obtener categorías para el select
    const categories = await Category.findAll({
      order: [['name', 'ASC']]
    });
    
    // Obtener etiquetas para el select múltiple
    const tags = await Tag.findAll({
      order: [['name', 'ASC']]
    });
    
    res.render('products/create-ai-tool', {
      title: 'Crear Herramienta de IA',
      categories,
      tags,
      errors: [],
      data: {},
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar el formulario',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Eliminar voto de un producto
exports.removeVote = async (req, res) => {
  try {
    const { id } = req.params;
    const user = JSON.parse(req.cookies.user);
    
    // Verificar si el producto existe
    const product = await Product.findByPk(id);
    
    if (!product) {
      return res.status(404).json({ success: false, message: 'Producto no encontrado' });
    }
    
    // Verificar si el usuario ha votado
    const existingVote = await Vote.findOne({
      where: {
        userId: user.id,
        productId: id
      }
    });
    
    if (!existingVote) {
      return res.status(400).json({ success: false, message: 'No has votado por este producto' });
    }
    
    // Eliminar voto y actualizar contador
    await sequelize.transaction(async (t) => {
      // Eliminar el voto
      await existingVote.destroy({ transaction: t });
      
      // Actualizar contador de votos
      await Product.decrement('votesCount', {
        by: 1,
        where: { id },
        transaction: t
      });
    });
    
    return res.status(200).json({ success: true, message: 'Voto eliminado correctamente' });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ success: false, message: 'Error al eliminar el voto' });
  }
};

// Eliminar un producto
exports.deleteProduct = async (req, res) => {
  try {
    const { id } = req.params;
    const user = JSON.parse(req.cookies.user);
    
    // Verificar si el producto existe
    const product = await Product.findByPk(id);
    
    if (!product) {
      return res.status(404).json({ success: false, message: 'Producto no encontrado' });
    }
    
    // Verificar si el usuario es el creador o un administrador
    if (product.userId !== user.id && user.role !== 'admin') {
      return res.status(403).json({ success: false, message: 'No tienes permiso para eliminar este producto' });
    }
    
    // Eliminar imagen asociada
    if (product.image) {
      const imagePath = path.join(__dirname, '../public/uploads/products', product.image);
      if (fs.existsSync(imagePath)) {
        fs.unlinkSync(imagePath);
      }
    }
    
    // Eliminar el producto
    await product.destroy();
    
    return res.status(200).json({ success: true, message: 'Producto eliminado correctamente' });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ success: false, message: 'Error al eliminar el producto' });
  }
};

module.exports = exports; 