const { User, Product, Vote, Category, Tag } = require('../models');

// Mostrar perfil de usuario
exports.getProfile = async (req, res) => {
  try {
    const { id } = req.params;
    
    // Obtener información del usuario
    const user = await User.findByPk(id, {
      attributes: ['id', 'name', 'email', 'avatar', 'createdAt', 'about']
    });
    
    if (!user) {
      return res.status(404).render('404', {
        title: 'Usuario no encontrado',
        user: req.cookies.user ? JSON.parse(req.cookies.user) : null
      });
    }
    
    // Obtener productos publicados por el usuario
    const products = await Product.findAll({
      where: { userId: id },
      include: [
        { model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] }
      ],
      order: [['createdAt', 'DESC']],
      limit: 10
    });
    
    // Obtener productos votados por el usuario
    const votes = await Vote.findAll({
      where: { userId: id },
      include: [
        { 
          model: Product,
          include: [
            { model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] }
          ]
        }
      ],
      order: [['createdAt', 'DESC']],
      limit: 10
    });
    
    const votedProducts = votes.map(vote => vote.Product);
    
    res.render('profile', {
      title: `Perfil de ${user.name}`,
      profileUser: user,
      products,
      votedProducts,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar el perfil',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Mostrar formulario de edición de perfil
exports.showEditForm = async (req, res) => {
  try {
    // Verificar si el usuario está autenticado
    if (!req.cookies.user) {
      return res.redirect('/auth/login');
    }
    
    const loggedUser = JSON.parse(req.cookies.user);
    
    // Obtener información del usuario
    const user = await User.findByPk(loggedUser.id);
    
    if (!user) {
      return res.status(404).render('404', {
        title: 'Usuario no encontrado',
        user: req.cookies.user ? JSON.parse(req.cookies.user) : null
      });
    }
    
    res.render('profile/edit', {
      title: 'Editar Perfil',
      profileUser: user,
      errors: [],
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar el formulario de edición',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Actualizar perfil de usuario
exports.updateProfile = async (req, res) => {
  try {
    // Verificar si el usuario está autenticado
    if (!req.cookies.user) {
      return res.redirect('/auth/login');
    }
    
    const loggedUser = JSON.parse(req.cookies.user);
    
    // Obtener información del usuario
    const user = await User.findByPk(loggedUser.id);
    
    if (!user) {
      return res.status(404).render('404', {
        title: 'Usuario no encontrado',
        user: req.cookies.user ? JSON.parse(req.cookies.user) : null
      });
    }
    
    // Actualizar datos del usuario
    const { name, email, about } = req.body;
    
    await user.update({
      name,
      email,
      about
    });
    
    // Si se subió una nueva imagen de perfil
    if (req.file) {
      await user.update({
        avatar: req.file.filename
      });
    }
    
    // Actualizar cookie con los datos actualizados
    const updatedUser = {
      id: user.id,
      name: user.name,
      email: user.email,
      avatar: user.avatar,
      isAdmin: user.isAdmin
    };
    
    res.cookie('user', JSON.stringify(updatedUser), {
      httpOnly: true,
      maxAge: 24 * 60 * 60 * 1000 // 1 día
    });
    
    req.flash('success', 'Perfil actualizado correctamente');
    res.redirect(`/profile/${user.id}`);
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al actualizar el perfil',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Obtener mis productos
exports.getMyProducts = async (req, res) => {
  try {
    // Verificar si el usuario está autenticado
    if (!req.cookies.user) {
      return res.redirect('/auth/login');
    }
    
    const loggedUser = JSON.parse(req.cookies.user);
    
    // Obtener productos publicados por el usuario
    const products = await Product.findAll({
      where: { userId: loggedUser.id },
      include: [
        { model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] },
        { model: Category, as: 'category', attributes: ['id', 'name', 'slug'] }
      ],
      order: [['createdAt', 'DESC']]
    });
    
    res.render('profile/my-products', {
      title: 'Mis Productos',
      products,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar tus productos',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Obtener mis votos
exports.getMyVotes = async (req, res) => {
  try {
    // Verificar si el usuario está autenticado
    if (!req.cookies.user) {
      return res.redirect('/auth/login');
    }
    
    const loggedUser = JSON.parse(req.cookies.user);
    
    // Obtener productos votados por el usuario
    const votes = await Vote.findAll({
      where: { userId: loggedUser.id },
      include: [
        { 
          model: Product,
          include: [
            { model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] },
            { model: Category, as: 'category', attributes: ['id', 'name', 'slug'] }
          ]
        }
      ],
      order: [['createdAt', 'DESC']]
    });
    
    const products = votes.map(vote => vote.Product);
    
    res.render('profile/my-votes', {
      title: 'Mis Votos',
      products,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar tus votos',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
}; 