const { Product, User, Category, Tag } = require('../models');
const { Op } = require('sequelize');

// Controlador para la búsqueda avanzada
exports.advancedSearch = async (req, res) => {
  try {
    const { query, category, sortBy } = req.query;
    const page = parseInt(req.query.page) || 1;
    const limit = 10;
    const offset = (page - 1) * limit;
    
    // Construir condiciones de búsqueda
    const whereConditions = {};
    
    // Búsqueda por texto
    if (query) {
      whereConditions[Op.or] = [
        { name: { [Op.like]: `%${query}%` } },
        { description: { [Op.like]: `%${query}%` } }
      ];
    }
    
    // Filtrar por categoría
    if (category) {
      whereConditions.categoryId = category;
    }
    
    // Determinar orden
    let order;
    if (sortBy === 'votes') {
      order = [['votesCount', 'DESC']];
    } else if (sortBy === 'name') {
      order = [['name', 'ASC']];
    } else {
      order = [['createdAt', 'DESC']]; // Por defecto, ordenar por fecha
    }
    
    // Obtener categorías para los filtros
    const categories = await Category.findAll({
      order: [['name', 'ASC']]
    });
    
    // Realizar la búsqueda
    const { count, rows: products } = await Product.findAndCountAll({
      where: whereConditions,
      include: [
        { model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] },
        { model: Category, as: 'category', attributes: ['id', 'name', 'slug'] }
      ],
      order,
      limit,
      offset
    });
    
    const totalPages = Math.ceil(count / limit);
    
    res.render('search', {
      title: 'Búsqueda Avanzada',
      products,
      categories,
      query: query || '',
      selectedCategory: category || '',
      sortBy: sortBy || 'date',
      currentPage: page,
      totalPages,
      count,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al realizar la búsqueda',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
}; 