const { Tag, Product, User, sequelize } = require('../models');
const { Op } = require('sequelize');
const slugify = require('slugify');
const { validationResult } = require('express-validator');

// Mostrar todas las etiquetas
exports.getTags = async (req, res) => {
  try {
    // Obtener todas las etiquetas con conteo de productos
    const tags = await Tag.findAll({
      include: [
        {
          model: Product,
          as: 'products',
          attributes: ['id'],
          through: { attributes: [] }
        }
      ],
      order: [['name', 'ASC']]
    });
    
    // Formatear datos para la vista
    const tagsWithCount = tags.map(tag => {
      const count = tag.products ? tag.products.length : 0;
      return {
        ...tag.toJSON(),
        productCount: count
      };
    });
    
    res.render('admin/tags/index', {
      title: 'Gestión de Etiquetas',
      layout: 'layouts/admin',
      path: '/admin/tags',
      tags: tagsWithCount,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar las etiquetas',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Mostrar productos por etiqueta
exports.getTagProducts = async (req, res) => {
  try {
    const { slug } = req.params;
    const page = parseInt(req.query.page) || 1;
    const limit = 10;
    const offset = (page - 1) * limit;
    const sortBy = req.query.sort || 'date'; // 'date' o 'votes'
    
    // Obtener la etiqueta
    const tag = await Tag.findOne({
      where: { slug }
    });
    
    if (!tag) {
      return res.status(404).render('404', {
        title: 'Etiqueta no encontrada',
        user: req.cookies.user ? JSON.parse(req.cookies.user) : null
      });
    }
    
    let order;
    if (sortBy === 'votes') {
      order = [['votesCount', 'DESC']];
    } else {
      order = [['createdAt', 'DESC']];
    }
    
    // Obtener productos por etiqueta con paginación
    const { count, rows: products } = await Product.findAndCountAll({
      include: [
        { model: User, as: 'creator', attributes: ['id', 'name', 'avatar'] },
        {
          model: Tag,
          as: 'tags',
          where: { id: tag.id },
          through: { attributes: [] },
          attributes: ['id', 'name', 'slug']
        }
      ],
      order,
      distinct: true,
      limit,
      offset
    });
    
    const totalPages = Math.ceil(count / limit);
    
    res.render('tags/show', {
      title: `Etiqueta: ${tag.name}`,
      tag,
      products,
      currentPage: page,
      totalPages,
      sortBy,
      count,
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar los productos por etiqueta',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Admin: Mostrar formulario para crear etiqueta
exports.showCreateForm = async (req, res) => {
  res.render('admin/tags/create', {
    title: 'Crear Etiqueta',
    layout: 'layouts/admin',
    path: '/admin/tags',
    errors: [],
    user: req.cookies.user ? JSON.parse(req.cookies.user) : null
  });
};

// Admin: Crear etiqueta
exports.createTag = async (req, res) => {
  // Verificar errores de validación
  const errors = validationResult(req);
  
  if (!errors.isEmpty()) {
    return res.status(400).render('admin/tags/create', {
      title: 'Crear Etiqueta',
      layout: 'layouts/admin',
      path: '/admin/tags',
      errors: errors.array(),
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
  
  try {
    const { name } = req.body;
    
    // Crear slug
    const slug = slugify(name, {
      lower: true,
      strict: true
    });
    
    // Verificar si ya existe una etiqueta con ese slug
    const existingTag = await Tag.findOne({
      where: { slug }
    });
    
    if (existingTag) {
      return res.status(400).render('admin/tags/create', {
        title: 'Crear Etiqueta',
        layout: 'layouts/admin',
        path: '/admin/tags',
        errors: [{ msg: 'Ya existe una etiqueta con ese nombre' }],
        user: req.cookies.user ? JSON.parse(req.cookies.user) : null
      });
    }
    
    // Crear etiqueta
    await Tag.create({
      name,
      slug
    });
    
    req.flash('success', 'Etiqueta creada correctamente');
    res.redirect('/admin/tags');
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al crear la etiqueta',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Admin: Mostrar formulario para editar etiqueta
exports.showEditForm = async (req, res) => {
  try {
    const { id } = req.params;
    
    const tag = await Tag.findByPk(id);
    
    if (!tag) {
      return res.status(404).render('404', {
        title: 'Etiqueta no encontrada',
        user: req.cookies.user ? JSON.parse(req.cookies.user) : null
      });
    }
    
    res.render('admin/tags/edit', {
      title: 'Editar Etiqueta',
      layout: 'layouts/admin',
      path: '/admin/tags',
      tag,
      errors: [],
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al cargar el formulario de edición',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Admin: Actualizar etiqueta
exports.updateTag = async (req, res) => {
  // Verificar errores de validación
  const errors = validationResult(req);
  
  if (!errors.isEmpty()) {
    return res.status(400).render('admin/tags/edit', {
      title: 'Editar Etiqueta',
      tag: { ...req.body, id: req.params.id },
      errors: errors.array(),
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
  
  try {
    const { id } = req.params;
    const { name } = req.body;
    
    const tag = await Tag.findByPk(id);
    
    if (!tag) {
      return res.status(404).render('404', {
        title: 'Etiqueta no encontrada',
        user: req.cookies.user ? JSON.parse(req.cookies.user) : null
      });
    }
    
    // Crear slug
    const slug = slugify(name, {
      lower: true,
      strict: true
    });
    
    // Verificar si ya existe otra etiqueta con ese slug
    const existingTag = await Tag.findOne({
      where: {
        slug,
        id: { [Op.ne]: id }
      }
    });
    
    if (existingTag) {
      return res.status(400).render('admin/tags/edit', {
        title: 'Editar Etiqueta',
        tag: { ...tag.toJSON(), name },
        errors: [{ msg: 'Ya existe una etiqueta con ese nombre' }],
        user: req.cookies.user ? JSON.parse(req.cookies.user) : null
      });
    }
    
    // Actualizar etiqueta
    await tag.update({
      name,
      slug
    });
    
    req.flash('success', 'Etiqueta actualizada correctamente');
    res.redirect('/admin/tags');
  } catch (error) {
    console.error(error);
    res.status(500).render('error', {
      title: 'Error',
      message: 'Error al actualizar la etiqueta',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Admin: Eliminar etiqueta
exports.deleteTag = async (req, res) => {
  try {
    const { id } = req.params;
    
    const tag = await Tag.findByPk(id);
    
    if (!tag) {
      return res.status(404).json({ success: false, message: 'Etiqueta no encontrada' });
    }
    
    await tag.destroy();
    
    res.json({ success: true, message: 'Etiqueta eliminada correctamente' });
  } catch (error) {
    console.error(error);
    res.status(500).json({ success: false, message: 'Error al eliminar la etiqueta' });
  }
}; 