-- Crear la base de datos
CREATE DATABASE IF NOT EXISTS product_hunt_clone CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

-- Usar la base de datos
USE product_hunt_clone;

-- Tabla de usuarios
CREATE TABLE IF NOT EXISTS Users (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(100) NOT NULL,
  email VARCHAR(100) NOT NULL UNIQUE,
  password VARCHAR(255) NOT NULL,
  avatar VARCHAR(255) DEFAULT 'default.png',
  isAdmin BOOLEAN DEFAULT false,
  createdAt DATETIME NOT NULL,
  updatedAt DATETIME NOT NULL
);

-- Tabla de categorías
CREATE TABLE IF NOT EXISTS Categories (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(100) NOT NULL UNIQUE,
  slug VARCHAR(100) NOT NULL UNIQUE,
  description TEXT,
  icon VARCHAR(100) DEFAULT 'fa-solid fa-folder',
  createdAt DATETIME NOT NULL,
  updatedAt DATETIME NOT NULL
);

-- Tabla de productos
CREATE TABLE IF NOT EXISTS Products (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(100) NOT NULL,
  description TEXT NOT NULL,
  url VARCHAR(255) NOT NULL,
  image VARCHAR(255) NOT NULL,
  votesCount INT DEFAULT 0,
  userId INT NOT NULL,
  categoryId INT,
  createdAt DATETIME NOT NULL,
  updatedAt DATETIME NOT NULL,
  FOREIGN KEY (userId) REFERENCES Users(id) ON DELETE CASCADE,
  FOREIGN KEY (categoryId) REFERENCES Categories(id) ON DELETE SET NULL
);

-- Tabla de comentarios
CREATE TABLE IF NOT EXISTS Comments (
  id INT AUTO_INCREMENT PRIMARY KEY,
  text TEXT NOT NULL,
  userId INT NOT NULL,
  productId INT NOT NULL,
  createdAt DATETIME NOT NULL,
  updatedAt DATETIME NOT NULL,
  FOREIGN KEY (userId) REFERENCES Users(id) ON DELETE CASCADE,
  FOREIGN KEY (productId) REFERENCES Products(id) ON DELETE CASCADE
);

-- Tabla de votos
CREATE TABLE IF NOT EXISTS Votes (
  id INT AUTO_INCREMENT PRIMARY KEY,
  userId INT NOT NULL,
  productId INT NOT NULL,
  createdAt DATETIME NOT NULL,
  updatedAt DATETIME NOT NULL,
  UNIQUE KEY unique_vote (userId, productId),
  FOREIGN KEY (userId) REFERENCES Users(id) ON DELETE CASCADE,
  FOREIGN KEY (productId) REFERENCES Products(id) ON DELETE CASCADE
);

-- Tabla de notificaciones
CREATE TABLE IF NOT EXISTS Notifications (
  id INT AUTO_INCREMENT PRIMARY KEY,
  type VARCHAR(50) NOT NULL,
  message TEXT NOT NULL,
  link VARCHAR(255),
  isRead BOOLEAN DEFAULT false,
  userId INT NOT NULL,
  createdAt DATETIME NOT NULL,
  updatedAt DATETIME NOT NULL,
  FOREIGN KEY (userId) REFERENCES Users(id) ON DELETE CASCADE
);

-- Tabla de etiquetas
CREATE TABLE IF NOT EXISTS Tags (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(50) NOT NULL UNIQUE,
  slug VARCHAR(50) NOT NULL UNIQUE,
  createdAt DATETIME NOT NULL,
  updatedAt DATETIME NOT NULL
);

-- Tabla de relación entre productos y etiquetas
CREATE TABLE IF NOT EXISTS ProductTags (
  id INT AUTO_INCREMENT PRIMARY KEY,
  productId INT NOT NULL,
  tagId INT NOT NULL,
  UNIQUE KEY unique_product_tag (productId, tagId),
  FOREIGN KEY (productId) REFERENCES Products(id) ON DELETE CASCADE,
  FOREIGN KEY (tagId) REFERENCES Tags(id) ON DELETE CASCADE
);

-- Insertar usuario de prueba (contraseña: password123)
INSERT INTO Users (name, email, password, isAdmin, createdAt, updatedAt)
VALUES ('Usuario Demo', 'demo@example.com', '$2a$10$X/4iyFq7gFhG9XY1.7EjdOvGfGD.dS6R.ueGI1hbvFiANgvz0pKXm', true, NOW(), NOW());

-- Insertar categorías de prueba
INSERT INTO Categories (name, slug, description, icon, createdAt, updatedAt)
VALUES 
('Productividad', 'productividad', 'Herramientas para mejorar tu productividad', 'fa-solid fa-tasks', NOW(), NOW()),
('Marketing', 'marketing', 'Herramientas de marketing y crecimiento', 'fa-solid fa-bullhorn', NOW(), NOW()),
('Diseño', 'diseno', 'Herramientas y recursos para diseñadores', 'fa-solid fa-palette', NOW(), NOW()),
('Desarrollo', 'desarrollo', 'Herramientas para desarrolladores', 'fa-solid fa-code', NOW(), NOW()),
('Finanzas', 'finanzas', 'Aplicaciones y servicios financieros', 'fa-solid fa-dollar-sign', NOW(), NOW());

-- Insertar etiquetas de prueba
INSERT INTO Tags (name, slug, createdAt, updatedAt)
VALUES 
('JavaScript', 'javascript', NOW(), NOW()),
('Open Source', 'open-source', NOW(), NOW()),
('SaaS', 'saas', NOW(), NOW()),
('AI', 'ai', NOW(), NOW()),
('Freemium', 'freemium', NOW(), NOW()),
('Mobile', 'mobile', NOW(), NOW()),
('Web App', 'web-app', NOW(), NOW());

-- Insertar productos de prueba
INSERT INTO Products (name, description, url, image, userId, categoryId, createdAt, updatedAt)
VALUES 
('Producto Demo 1', 'Este es un producto de demostración para probar la aplicación.', 'https://example.com/demo1', 'default-product.png', 1, 1, NOW(), NOW()),
('Producto Demo 2', 'Otro producto de demostración con una descripción más larga para ver cómo se ve en la interfaz.', 'https://example.com/demo2', 'default-product.png', 1, 2, NOW(), NOW());

-- Asignar etiquetas a productos de prueba
INSERT INTO ProductTags (productId, tagId)
VALUES 
(1, 3), -- Producto 1 - SaaS
(1, 5), -- Producto 1 - Freemium
(1, 7), -- Producto 1 - Web App
(2, 4), -- Producto 2 - AI
(2, 7); -- Producto 2 - Web App 