const jwt = require('jsonwebtoken');
const { User } = require('../models');

// Middleware para proteger rutas que requieren autenticación
exports.isAuthenticated = async (req, res, next) => {
  try {
    // Verificar si existe el token
    const token = req.cookies.token;
    
    if (!token) {
      return res.status(401).redirect('/auth/login');
    }
    
    // Verificar el token
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    
    // Buscar el usuario
    const user = await User.findByPk(decoded.id);
    
    if (!user) {
      res.clearCookie('token');
      res.clearCookie('user');
      return res.status(401).redirect('/auth/login');
    }
    
    // Guardar el usuario en req
    req.user = user;
    
    next();
  } catch (error) {
    console.error(error);
    res.clearCookie('token');
    res.clearCookie('user');
    res.status(401).redirect('/auth/login');
  }
};

// Middleware para verificar si el usuario es administrador
exports.isAdmin = async (req, res, next) => {
  try {
    // Verificar si existe el token
    const token = req.cookies.token;
    
    if (!token) {
      return res.status(401).redirect('/auth/login');
    }
    
    // Verificar el token
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    
    // Buscar el usuario
    const user = await User.findByPk(decoded.id);
    
    if (!user || !user.isAdmin) {
      return res.status(403).render('error', {
        title: 'Acceso Denegado',
        message: 'No tienes permisos para acceder a esta sección',
        user: req.cookies.user ? JSON.parse(req.cookies.user) : null
      });
    }
    
    // Guardar el usuario en req
    req.user = user;
    
    // Agregar la variable path para las vistas de administración
    res.locals.path = req.originalUrl;
    
    next();
  } catch (error) {
    console.error(error);
    res.status(403).render('error', {
      title: 'Acceso Denegado',
      message: 'Error al verificar permisos',
      user: req.cookies.user ? JSON.parse(req.cookies.user) : null
    });
  }
};

// Middleware para verificar si el usuario está autenticado (opcional)
exports.checkAuth = async (req, res, next) => {
  try {
    // Verificar si existe el token
    const token = req.cookies.token;
    
    if (!token) {
      req.user = null;
      return next();
    }
    
    // Verificar el token
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    
    // Buscar el usuario
    const user = await User.findByPk(decoded.id);
    
    if (!user) {
      req.user = null;
      res.clearCookie('token');
      res.clearCookie('user');
      return next();
    }
    
    // Guardar el usuario en req
    req.user = user;
    
    next();
  } catch (error) {
    req.user = null;
    res.clearCookie('token');
    res.clearCookie('user');
    next();
  }
}; 