// JavaScript para el panel de administración

document.addEventListener('DOMContentLoaded', function() {
  // Asegurarse de que el CSS de corrección se haya cargado
  ensureCssLoaded('/css/admin-fix.css');
  
  // Agregar botón de toggle para sidebar en móvil si no existe
  const adminHeader = document.querySelector('.admin-header');
  if (adminHeader && !document.getElementById('toggleSidebar')) {
    const toggleBtn = document.createElement('button');
    toggleBtn.id = 'toggleSidebar';
    toggleBtn.className = 'btn btn-link d-lg-none me-2';
    toggleBtn.innerHTML = '<i class="fa-solid fa-bars"></i>';
    adminHeader.querySelector('.d-flex').prepend(toggleBtn);
  }
  
  // Toggle para sidebar en móvil
  const toggleSidebarBtn = document.getElementById('toggleSidebar');
  if (toggleSidebarBtn) {
    toggleSidebarBtn.addEventListener('click', function() {
      document.querySelector('.admin-sidebar').classList.toggle('show');
    });
  }
  
  // Cerrar sidebar al hacer clic fuera de él en móvil
  document.addEventListener('click', function(event) {
    const sidebar = document.querySelector('.admin-sidebar');
    const toggleBtn = document.getElementById('toggleSidebar');
    
    if (sidebar && sidebar.classList.contains('show') && 
        !sidebar.contains(event.target) && 
        toggleBtn && !toggleBtn.contains(event.target)) {
      sidebar.classList.remove('show');
    }
  });
  
  // Inicializar tooltips de Bootstrap
  const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
  tooltipTriggerList.map(function (tooltipTriggerEl) {
    return new bootstrap.Tooltip(tooltipTriggerEl);
  });
  
  // Inicializar confirmaciones de eliminación
  document.querySelectorAll('.delete-confirm').forEach(button => {
    button.addEventListener('click', function(e) {
      if (!confirm('¿Estás seguro de que deseas eliminar este elemento?')) {
        e.preventDefault();
      }
    });
  });
  
  // Inicializar gráficos si existen los elementos
  initCharts();
  
  // Asegurarse de que no haya superposición de texto
  fixTextOverlap();
  
  // Aplicar correcciones cuando la ventana cambie de tamaño
  window.addEventListener('resize', fixTextOverlap);
  
  // Aplicar correcciones cuando se cargue completamente la página
  window.addEventListener('load', fixTextOverlap);
});

// Función para asegurarse de que el CSS de corrección se haya cargado
function ensureCssLoaded(cssPath) {
  let cssLoaded = false;
  
  // Verificar si el CSS ya está cargado
  const links = document.getElementsByTagName('link');
  for (let i = 0; i < links.length; i++) {
    if (links[i].href.includes(cssPath)) {
      cssLoaded = true;
      break;
    }
  }
  
  // Si no está cargado, agregarlo
  if (!cssLoaded) {
    const link = document.createElement('link');
    link.rel = 'stylesheet';
    link.href = cssPath;
    document.head.appendChild(link);
  }
  
  // Aplicar estilos críticos directamente
  const style = document.createElement('style');
  style.textContent = `
    .admin-header h1 {
      overflow: hidden !important;
      text-overflow: ellipsis !important;
      white-space: nowrap !important;
      max-width: 50% !important;
      margin: 0 !important;
      padding: 0 !important;
    }
    
    .admin-header .dropdown-toggle span {
      display: inline-block !important;
      max-width: 120px !important;
      overflow: hidden !important;
      text-overflow: ellipsis !important;
      white-space: nowrap !important;
    }
    
    .sidebar-header a span {
      display: inline-block !important;
      max-width: 150px !important;
      overflow: hidden !important;
      text-overflow: ellipsis !important;
      white-space: nowrap !important;
    }
    
    .admin-sidebar .nav-link i {
      width: 20px !important;
      text-align: center !important;
      margin-right: 8px !important;
    }
  `;
  document.head.appendChild(style);
}

// Función para corregir superposición de texto
function fixTextOverlap() {
  // Corregir el título en el header
  const headerTitle = document.querySelector('.admin-header h1');
  if (headerTitle) {
    headerTitle.style.overflow = 'hidden';
    headerTitle.style.textOverflow = 'ellipsis';
    headerTitle.style.whiteSpace = 'nowrap';
    headerTitle.style.maxWidth = '50%';
  }
  
  // Corregir el nombre de usuario en el dropdown
  const userDropdownText = document.querySelector('.admin-header .dropdown-toggle span');
  if (userDropdownText) {
    userDropdownText.style.maxWidth = '120px';
    userDropdownText.style.overflow = 'hidden';
    userDropdownText.style.textOverflow = 'ellipsis';
    userDropdownText.style.whiteSpace = 'nowrap';
  }
  
  // Corregir el logo en el sidebar
  const sidebarLogo = document.querySelector('.sidebar-header a span');
  if (sidebarLogo) {
    sidebarLogo.style.maxWidth = '150px';
    sidebarLogo.style.overflow = 'hidden';
    sidebarLogo.style.textOverflow = 'ellipsis';
    sidebarLogo.style.whiteSpace = 'nowrap';
  }
  
  // Corregir elementos del dropdown
  document.querySelectorAll('.admin-header .dropdown-item').forEach(item => {
    item.style.whiteSpace = 'nowrap';
    item.style.overflow = 'hidden';
    item.style.textOverflow = 'ellipsis';
  });
  
  // Asegurar que no haya superposición en los íconos del sidebar
  document.querySelectorAll('.admin-sidebar .nav-link i').forEach(icon => {
    icon.style.width = '20px';
    icon.style.textAlign = 'center';
    icon.style.marginRight = '8px';
  });
  
  // Asegurar que no haya superposición en el contenido principal
  document.querySelectorAll('.admin-main .card-title, .admin-main h5, .admin-main h6').forEach(heading => {
    heading.style.overflow = 'hidden';
    heading.style.textOverflow = 'ellipsis';
    heading.style.whiteSpace = 'nowrap';
  });
  
  // Eliminar cualquier texto superpuesto
  const overlayElements = document.querySelectorAll('.text-overlay');
  overlayElements.forEach(element => {
    element.style.display = 'none';
  });
}

// Función para inicializar gráficos
function initCharts() {
  // Gráfico de categorías
  const categoryChartEl = document.getElementById('categoryChart');
  if (categoryChartEl) {
    const ctx = categoryChartEl.getContext('2d');
    new Chart(ctx, {
      type: 'bar',
      data: {
        labels: categoryLabels || [],
        datasets: [{
          label: 'Productos por categoría',
          data: categoryData || [],
          backgroundColor: 'rgba(40, 167, 69, 0.6)',
          borderColor: 'rgba(40, 167, 69, 1)',
          borderWidth: 1
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        scales: {
          y: {
            beginAtZero: true,
            ticks: {
              precision: 0
            }
          }
        }
      }
    });
  }
  
  // Gráfico de usuarios
  const userChartEl = document.getElementById('userChart');
  if (userChartEl) {
    const ctx = userChartEl.getContext('2d');
    new Chart(ctx, {
      type: 'line',
      data: {
        labels: userLabels || [],
        datasets: [{
          label: 'Nuevos usuarios',
          data: userData || [],
          backgroundColor: 'rgba(0, 123, 255, 0.1)',
          borderColor: 'rgba(0, 123, 255, 1)',
          borderWidth: 2,
          tension: 0.3,
          fill: true
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        scales: {
          y: {
            beginAtZero: true,
            ticks: {
              precision: 0
            }
          }
        }
      }
    });
  }
  
  // Gráfico de productos
  const productChartEl = document.getElementById('productChart');
  if (productChartEl) {
    const ctx = productChartEl.getContext('2d');
    new Chart(ctx, {
      type: 'pie',
      data: {
        labels: productLabels || [],
        datasets: [{
          data: productData || [],
          backgroundColor: [
            'rgba(40, 167, 69, 0.6)',
            'rgba(0, 123, 255, 0.6)',
            'rgba(255, 193, 7, 0.6)',
            'rgba(220, 53, 69, 0.6)',
            'rgba(111, 66, 193, 0.6)'
          ],
          borderColor: [
            'rgba(40, 167, 69, 1)',
            'rgba(0, 123, 255, 1)',
            'rgba(255, 193, 7, 1)',
            'rgba(220, 53, 69, 1)',
            'rgba(111, 66, 193, 1)'
          ],
          borderWidth: 1
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
          legend: {
            position: 'bottom'
          }
        }
      }
    });
  }
}

// Función para manejar eliminaciones AJAX
function deleteItem(url, itemId, elementId) {
  if (confirm('¿Estás seguro de que deseas eliminar este elemento?')) {
    fetch(url + '/' + itemId, {
      method: 'DELETE',
      headers: {
        'Content-Type': 'application/json'
      }
    })
    .then(response => response.json())
    .then(data => {
      if (data.success) {
        // Eliminar elemento de la UI
        const element = document.getElementById(elementId);
        if (element) {
          element.remove();
        }
        
        // Mostrar mensaje de éxito
        showAlert('success', data.message);
      } else {
        showAlert('danger', data.message || 'Error al eliminar');
      }
    })
    .catch(error => {
      console.error('Error:', error);
      showAlert('danger', 'Error al procesar la solicitud');
    });
  }
}

// Función para mostrar alertas
function showAlert(type, message) {
  const alertContainer = document.getElementById('alertContainer');
  if (!alertContainer) return;
  
  const alert = document.createElement('div');
  alert.className = `alert alert-${type} alert-dismissible fade show`;
  alert.innerHTML = `
    ${message}
    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
  `;
  
  alertContainer.appendChild(alert);
  
  // Auto-cerrar después de 5 segundos
  setTimeout(() => {
    const bsAlert = new bootstrap.Alert(alert);
    bsAlert.close();
  }, 5000);
} 