document.addEventListener('DOMContentLoaded', function() {
  // Elementos DOM
  const notificationBadge = document.getElementById('notification-badge');
  const notificationsContainer = document.getElementById('notifications-container');
  const markAllReadBtn = document.getElementById('mark-all-read');
  
  // Cargar notificaciones al iniciar
  loadNotifications();
  
  // Configurar intervalo para actualizar notificaciones cada minuto
  setInterval(loadNotifications, 60000);
  
  // Evento para marcar todas como leídas
  if (markAllReadBtn) {
    markAllReadBtn.addEventListener('click', markAllAsRead);
  }
  
  // Función para cargar notificaciones
  function loadNotifications() {
    fetch('/notifications')
      .then(response => response.json())
      .then(data => {
        if (data.success) {
          updateNotificationBadge(data.unreadCount);
          renderNotifications(data.notifications);
        }
      })
      .catch(error => console.error('Error al cargar notificaciones:', error));
  }
  
  // Función para actualizar el badge de notificaciones
  function updateNotificationBadge(count) {
    if (count > 0) {
      notificationBadge.textContent = count > 9 ? '9+' : count;
      notificationBadge.classList.remove('d-none');
    } else {
      notificationBadge.classList.add('d-none');
    }
  }
  
  // Función para renderizar notificaciones
  function renderNotifications(notifications) {
    if (!notificationsContainer) return;
    
    if (notifications.length === 0) {
      notificationsContainer.innerHTML = `
        <div class="notification-empty">
          <i class="fa-solid fa-bell-slash fa-2x mb-3 text-muted"></i>
          <p class="mb-0">No tienes notificaciones</p>
        </div>
      `;
      return;
    }
    
    let html = '';
    
    notifications.forEach(notification => {
      const date = new Date(notification.createdAt);
      const formattedDate = date.toLocaleDateString() + ' ' + date.toLocaleTimeString();
      const unreadClass = notification.isRead ? '' : 'unread';
      let icon = 'fa-solid fa-bell';
      
      // Determinar icono según tipo de notificación
      if (notification.type === 'vote') {
        icon = 'fa-solid fa-thumbs-up';
      } else if (notification.type === 'comment') {
        icon = 'fa-solid fa-comment';
      }
      
      html += `
        <a href="${notification.link || '#'}" class="dropdown-item notification-item ${unreadClass}" data-id="${notification.id}">
          <div class="d-flex align-items-center">
            <div class="notification-icon ${notification.type} me-3">
              <i class="${icon}"></i>
            </div>
            <div class="flex-grow-1">
              <p class="notification-message mb-1">${notification.message}</p>
              <small class="notification-time">${formattedDate}</small>
            </div>
            ${!notification.isRead ? '<button class="btn btn-sm mark-as-read" data-id="' + notification.id + '"><i class="fa-solid fa-check"></i></button>' : ''}
          </div>
        </a>
      `;
    });
    
    notificationsContainer.innerHTML = html;
    
    // Añadir eventos a los botones de marcar como leído
    document.querySelectorAll('.mark-as-read').forEach(button => {
      button.addEventListener('click', function(e) {
        e.preventDefault();
        e.stopPropagation();
        const id = this.getAttribute('data-id');
        markAsRead(id);
      });
    });
    
    // Marcar como leída al hacer clic en la notificación
    document.querySelectorAll('.notification-item').forEach(item => {
      item.addEventListener('click', function() {
        const id = this.getAttribute('data-id');
        if (!this.classList.contains('unread')) return;
        markAsRead(id, false);
      });
    });
  }
  
  // Función para marcar una notificación como leída
  function markAsRead(id, reload = true) {
    fetch(`/notifications/${id}/read`, {
      method: 'POST'
    })
      .then(response => response.json())
      .then(data => {
        if (data.success && reload) {
          loadNotifications();
        }
      })
      .catch(error => console.error('Error al marcar notificación como leída:', error));
  }
  
  // Función para marcar todas las notificaciones como leídas
  function markAllAsRead() {
    fetch('/notifications/read-all', {
      method: 'POST'
    })
      .then(response => response.json())
      .then(data => {
        if (data.success) {
          loadNotifications();
        }
      })
      .catch(error => console.error('Error al marcar todas las notificaciones como leídas:', error));
  }
}); 