const express = require('express');
const router = express.Router();
const { check } = require('express-validator');
const adminController = require('../controllers/adminController');
const categoryController = require('../controllers/categoryController');
const tagController = require('../controllers/tagController');
const { checkAuth, isAdmin } = require('../middlewares/auth');

// Middleware para proteger todas las rutas de admin
router.use(checkAuth, isAdmin);

// Dashboard principal
router.get('/', adminController.dashboard);

// Rutas de estadísticas
router.get('/stats', adminController.stats);

// Rutas de usuarios
router.get('/users', adminController.getUsers);
router.get('/users/:id', adminController.getUser);
router.post('/users/:id', adminController.updateUser);
router.delete('/users/:id', adminController.deleteUser);

// Rutas de productos
router.get('/products', adminController.getProducts);
router.get('/products/:id', adminController.getProduct);
router.post('/products/:id', adminController.updateProduct);
router.delete('/products/:id', adminController.deleteProduct);

// Rutas de categorías
router.get('/categories', categoryController.adminGetCategories);

router.get('/categories/create', categoryController.adminShowCreateForm);
router.post('/categories/create', [
  check('name').notEmpty().withMessage('El nombre es obligatorio'),
], categoryController.adminCreateCategory);

router.get('/categories/:id/edit', categoryController.adminShowEditForm);
router.post('/categories/:id', [
  check('name').notEmpty().withMessage('El nombre es obligatorio'),
], categoryController.adminUpdateCategory);

router.delete('/categories/:id', categoryController.adminDeleteCategory);

// Rutas de etiquetas
router.get('/tags', tagController.getTags);

router.get('/tags/create', tagController.showCreateForm);
router.post('/tags/create', [
  check('name').notEmpty().withMessage('El nombre es obligatorio'),
], tagController.createTag);

router.get('/tags/:id/edit', tagController.showEditForm);
router.post('/tags/:id', [
  check('name').notEmpty().withMessage('El nombre es obligatorio'),
], tagController.updateTag);

router.delete('/tags/:id', tagController.deleteTag);

// Rutas de notificaciones
router.get('/notifications', adminController.getNotifications);
router.post('/notifications', adminController.createNotification);
router.delete('/notifications/:id', adminController.deleteNotification);

module.exports = router; 