const express = require('express');
const router = express.Router();
const productController = require('../controllers/productController');
const { isAuthenticated, checkAuth } = require('../middlewares/auth');
const { uploadProductImage } = require('../middlewares/upload');
const { productValidator, commentValidator } = require('../middlewares/validators');

// Listar productos
router.get('/', checkAuth, productController.getProducts);

// Buscar productos
router.get('/search', checkAuth, productController.searchProducts);

// Formulario para crear producto
router.get('/create', isAuthenticated, productController.showCreateForm);

// Formulario para crear herramienta de IA
router.get('/create-ai-tool', isAuthenticated, productController.showCreateAIToolForm);

// Crear producto
router.post('/', 
  isAuthenticated, 
  uploadProductImage.single('image'), 
  productValidator, 
  productController.createProduct
);

// Productos de un usuario específico
router.get('/user/:userId', checkAuth, productController.getUserProducts);

// Votos de un usuario específico
router.get('/votes/user/:userId', checkAuth, productController.getUserVotes);

// Ver producto específico
router.get('/:id', checkAuth, productController.getProduct);

// Votar por un producto
router.post('/:id/vote', isAuthenticated, productController.voteProduct);

// Eliminar voto de un producto
router.delete('/:id/vote', isAuthenticated, productController.removeVote);

// Comentar un producto
router.post('/:id/comment', 
  isAuthenticated, 
  commentValidator, 
  productController.commentProduct
);

// Eliminar un producto
router.delete('/:id', isAuthenticated, productController.deleteProduct);

module.exports = router; 