const express = require('express');
const router = express.Router();
const profileController = require('../controllers/profileController');
const { checkAuth, isAuthenticated } = require('../middlewares/auth');
const multer = require('multer');
const path = require('path');

// Configurar multer para la subida de avatares
const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    cb(null, path.join(__dirname, '../public/uploads/avatars'));
  },
  filename: function (req, file, cb) {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    const ext = path.extname(file.originalname);
    cb(null, 'avatar-' + uniqueSuffix + ext);
  }
});

const upload = multer({ 
  storage: storage,
  limits: { fileSize: 5 * 1024 * 1024 }, // 5MB
  fileFilter: function (req, file, cb) {
    const filetypes = /jpeg|jpg|png|gif/;
    const mimetype = filetypes.test(file.mimetype);
    const extname = filetypes.test(path.extname(file.originalname).toLowerCase());
    
    if (mimetype && extname) {
      return cb(null, true);
    }
    
    cb(new Error('Solo se permiten archivos de imagen (jpg, jpeg, png, gif)'));
  }
});

// Rutas de perfil
// Mis productos
router.get('/products', isAuthenticated, profileController.getMyProducts);

// Mis votos
router.get('/votes', isAuthenticated, profileController.getMyVotes);

// La ruta de edición debe ir antes que la ruta dinámica para evitar conflictos
router.get('/edit', isAuthenticated, profileController.showEditForm);
router.post('/edit', isAuthenticated, upload.single('avatar'), profileController.updateProfile);

// Ver perfil de usuario
router.get('/:id', checkAuth, profileController.getProfile);

module.exports = router; 