const { sequelize } = require('./models/db');
const User = require('./models/User');
const Product = require('./models/Product');
const Category = require('./models/Category');
const Tag = require('./models/Tag');
const Comment = require('./models/Comment');
const Vote = require('./models/Vote');
const Notification = require('./models/Notification');
const ProductTag = require('./models/ProductTag');

// Función para sincronizar la base de datos
async function syncDatabase() {
  try {
    // Sincronizar todos los modelos con la base de datos
    // force: true eliminará las tablas existentes y las creará de nuevo
    await sequelize.sync({ force: true });
    
    console.log('Base de datos sincronizada correctamente.');
    
    // Insertar datos de prueba
    // Usuario administrador
    const admin = await User.create({
      name: 'Admin',
      email: 'admin@nodolatam.dev',
      password: '$2a$10$X/4iyFq7gFhG9XY1.7EjdOvGfGD.dS6R.ueGI1hbvFiANgvz0pKXm', // password123
      avatar: 'default.png',
      isAdmin: true
    });
    
    // Categorías
    const categories = await Category.bulkCreate([
      {
        name: 'Desarrollo',
        slug: 'desarrollo',
        description: 'Herramientas, librerías y recursos para desarrolladores',
        icon: 'fa-solid fa-code'
      },
      {
        name: 'Diseño',
        slug: 'diseno',
        description: 'Herramientas y recursos para diseñadores',
        icon: 'fa-solid fa-palette'
      },
      {
        name: 'Marketing',
        slug: 'marketing',
        description: 'Herramientas de marketing y crecimiento',
        icon: 'fa-solid fa-bullhorn'
      },
      {
        name: 'Productividad',
        slug: 'productividad',
        description: 'Aplicaciones y herramientas para mejorar la productividad',
        icon: 'fa-solid fa-tasks'
      },
      {
        name: 'Finanzas',
        slug: 'finanzas',
        description: 'Aplicaciones y servicios financieros',
        icon: 'fa-solid fa-dollar-sign'
      }
    ]);
    
    // Etiquetas
    const tags = await Tag.bulkCreate([
      { name: 'JavaScript', slug: 'javascript' },
      { name: 'Open Source', slug: 'open-source' },
      { name: 'SaaS', slug: 'saas' },
      { name: 'AI', slug: 'ai' },
      { name: 'Freemium', slug: 'freemium' },
      { name: 'Mobile', slug: 'mobile' },
      { name: 'Web App', slug: 'web-app' }
    ]);
    
    // Productos
    const products = await Product.bulkCreate([
      {
        name: 'Nodolatam Editor',
        description: 'Un editor de código en línea para desarrolladores latinoamericanos.',
        url: 'https://editor.nodolatam.dev',
        image: 'default-product.png',
        userId: admin.id,
        categoryId: categories[0].id
      },
      {
        name: 'Diseño Latino',
        description: 'Plataforma de recursos de diseño para profesionales latinoamericanos.',
        url: 'https://diseno.nodolatam.dev',
        image: 'default-product.png',
        userId: admin.id,
        categoryId: categories[1].id
      }
    ]);
    
    // Asignar etiquetas a productos
    await ProductTag.bulkCreate([
      { productId: products[0].id, tagId: tags[0].id }, // JavaScript
      { productId: products[0].id, tagId: tags[1].id }, // Open Source
      { productId: products[0].id, tagId: tags[6].id }, // Web App
      { productId: products[1].id, tagId: tags[4].id }, // Freemium
      { productId: products[1].id, tagId: tags[6].id }  // Web App
    ]);
    
    console.log('Datos de prueba insertados correctamente.');
    
    process.exit(0);
  } catch (error) {
    console.error('Error al sincronizar la base de datos:', error);
    process.exit(1);
  }
}

// Ejecutar la sincronización
syncDatabase(); 